/*
** ahead Nero - NRG file treating module
**
** Writen by Sakae Tatibana <tatibana@extra.hu>
**
** 2000, 8/3 coding start
**       
*/

#include <string.h>
#include "binary.h"

#define NRG_C
#include "nrg.h"

typedef struct {
	unsigned char mode;
	unsigned char track;
	unsigned char index;
	unsigned char unknown_a;
	unsigned char unknown_b;
	unsigned char mm;
	unsigned char ss;
	unsigned char ff;
}NRG_CUES_ELEMENT;

typedef struct {
	unsigned char ISRC[12];
	short block_size;
	int unknown_d;
	int offset_a;
	int offset_b;
	int track_end;
}NRG_DAOI_ELEMENT;

typedef struct {
	int offset;
	int length;
	int mode;
	int start;
	int unknown;
}NRG_ETNF_ELEMENT;

extern "C" int open_nrg_file(char *path, DAO_INFO *out);
int is_track_image_nrg(char *path);

static void read_cues(FILE *in, DAO_INFO *out);
static void read_daoi(FILE *in, DAO_INFO *out);
static void read_sinf(FILE *in, DAO_INFO *out);
static void read_etnf(FILE *in, DAO_INFO *out);

static void read_nrg_cues_element(FILE *in, NRG_CUES_ELEMENT *out);
static void read_nrg_daoi_element(FILE *in, NRG_DAOI_ELEMENT *out);
static void read_nrg_etnf_element(FILE *in, NRG_ETNF_ELEMENT *out);

static int bcd_msf_to_lbn(unsigned char mm, unsigned char ss, unsigned char ff);  

int open_nrg_file(char *path, DAO_INFO *out)
{
    int n;
	FILE *in;
	
	unsigned char pattern[4];

	static const unsigned char cues_marker[] = {
		'C', 'U', 'E', 'S',
	};

	static const unsigned char end_marker[] = {
		'E', 'N', 'D', '!',
	};

	static const unsigned char daoi_marker[] = {
		'D', 'A', 'O', 'I',
	};

	static const unsigned char sinf_marker[] = {
		'S', 'I', 'N', 'F',
	};

	static const unsigned char etnf_marker[] = {
		'E', 'T', 'N', 'F',
	};

	in = fopen(path, "rb");
	if(in == NULL){
		return 0;
	}

	out->path = path;
	
	fseek(in, -4, SEEK_END);
	read_be_int32(in, &n);
	fseek(in, n, SEEK_SET);

	fread(pattern, 1, 4, in);
	if(memcmp(pattern, cues_marker, 4)){
		goto OPEN_NRG_ERROR;
	}else{
		read_cues(in, out);
	}

	fread(pattern, 1, 4, in);
	if(memcmp(pattern, end_marker, 4) == 0){
		goto OPEN_NRG_SUCCESS;
	}

	if(memcmp(pattern, daoi_marker, 4)){
		goto OPEN_NRG_ERROR;
	}else{
		read_daoi(in, out);
	}
	
	while(fread(pattern, 1, 4, in)){
		if(memcmp(pattern, end_marker, 4) == 0){
			goto OPEN_NRG_SUCCESS;
		}else if(memcmp(pattern, sinf_marker, 4) == 0){
			read_sinf(in, out);
		}else if(memcmp(pattern, etnf_marker, 4) == 0){
			read_etnf(in, out);
		}else{
			goto OPEN_NRG_ERROR;
		}
	}
	
OPEN_NRG_SUCCESS:	
	fclose(in);
	return 1;
	
OPEN_NRG_ERROR:
	fclose(in);
	return 0;
}

extern int is_track_image_nrg(char *path)
{
	int n;
	FILE *in;

	unsigned char pattern[4];

	static const unsigned char marker[] = {
		0x54, 0x49, 0x4E, 0x46,
	};

	in = fopen(path, "rb");
	if(in == NULL){
		return 0;
	}

	fseek(in, -4, SEEK_END);
	read_be_int32(in, &n);
	fseek(in, n, SEEK_SET);

	fread(pattern, 1, 4, in);
	if(memcmp(pattern, marker, 4)){
		fclose(in);
		return 0;
	}

	fclose(in);
	return 1;
}

static void read_cues(FILE *in, DAO_INFO *out)
{
	int i,n;
	NRG_CUES_ELEMENT work;

	read_be_int32(in, &n);
	n /= 8;

	read_nrg_cues_element(in, &work); /* track - 0, index - 1 */
	read_nrg_cues_element(in, &work); /* track - 1, index - 0 */

	out->num_of_track = n / 2 - 1;

	for(i=0;i<n-2;i++){
		read_nrg_cues_element(in, &work); /* track - 1, index - 0 */
		if(work.index && work.track != 0xAA){
			out->track[i/2].index = i/2;
			strcpy(out->track[i/2].path, out->path);

			switch(work.mode){
			case 0x01:
				out->track[i/2].type = DAO_TRACK_TYPE_AUDIO;
				break;
			case 0x41:
				out->track[i/2].type = DAO_TRACK_TYPE_MODE1;
				break;
			case 0x61:
				out->track[i/2].type = DAO_TRACK_TYPE_MODE2;
				break;
			default:
				out->track[i/2].type = DAO_TRACK_TYPE_ELSE;
				out->num_of_track = i/2;
                return;
			}

			out->track[i/2].block_size = 2352;

			out->track[i/2].start = bcd_msf_to_lbn(work.mm, work.ss, work.ff);
			out->track[i/2].offset = (out->track[i/2].start) * 2352;
			out->track[i/2].start -= 150;
		}else{
			out->track[i/2].num_of_block = bcd_msf_to_lbn(work.mm, work.ss, work.ff);
			out->track[i/2].num_of_block -= out->track[i/2].start;
			out->track[i/2].num_of_block -= 150;
		}
	}

	return;
}

static void read_daoi(FILE *in, DAO_INFO *out)
{
	int i,n;

	char unknown[17];

	char last_track;

	NRG_DAOI_ELEMENT work;

	read_be_int32(in, &n);
	read_be_int32(in, &n);
	n -= 22; /* 4 + 17 + 1*/
	n /= 30;

    fread(unknown, 1, 17, in);
	last_track = (char)fgetc(in);

	if(n != last_track){
		/* the code has bug */
	}

	for(i=0;i<n;i++){
		read_nrg_daoi_element(in, &work);
		out->track[i].offset = work.offset_b;
		out->track[i].block_size = work.block_size;
	}
	
	return;
}

static void read_sinf(FILE *in, DAO_INFO *out)
{
	int n;
	char *buffer;

	read_be_int32(in, &n);

	buffer = (char *)malloc(n);
	fread(buffer, 1, n, in);
	free(buffer);

	return;
}

static void read_etnf(FILE *in, DAO_INFO *out)
{
	int i,n,t;
	NRG_ETNF_ELEMENT work;

	read_be_int32(in, &n);
	n /= 20;

	for(i=0;i<n;i++){
		read_nrg_etnf_element(in, &work);
		
		t = out->num_of_track;
		out->num_of_track += 1;
		
		out->track[t].index = t;

		strcpy(out->track[t].path, out->path);

		switch(work.mode){
		case 0:
			out->track[t].type = DAO_TRACK_TYPE_MODE1;
			out->track[t].block_size = 2048;
			break;
		case 1:
		case 2:
		case 3:
			out->track[t].type = DAO_TRACK_TYPE_MODE2;
			out->track[t].block_size = 2336;
			break;
		case 5:
			out->track[t].type = DAO_TRACK_TYPE_MODE1;
			out->track[t].block_size = 2352;
			break;
		case 6:
			out->track[t].type = DAO_TRACK_TYPE_MODE2;
			out->track[t].block_size = 2352;
			break;
		default:
			out->track[t].type = DAO_TRACK_TYPE_AUDIO;
			out->track[t].block_size = 2352;
		}

        out->track[t].offset = work.offset;

		out->track[t].start = work.start;
		out->track[t].num_of_block = (work.length / out->track[t].block_size) - 150;
	}

	return;
}

static void read_nrg_cues_element(FILE *in, NRG_CUES_ELEMENT *out)
{
	out->mode = (unsigned char)fgetc(in);
	out->track = (unsigned char)fgetc(in);
	out->index = (unsigned char)fgetc(in);
	out->unknown_a = (unsigned char)fgetc(in);
	out->unknown_b = (unsigned char)fgetc(in);
	out->mm = (unsigned char)fgetc(in);
	out->ss = (unsigned char)fgetc(in);
	out->ff = (unsigned char)fgetc(in);
}

static void read_nrg_daoi_element(FILE *in, NRG_DAOI_ELEMENT *out)
{
	fread(out->ISRC, 1, 12, in);

	read_be_int16(in, &(out->block_size));

	read_be_int32(in, &(out->unknown_d));

	read_be_int32(in, &(out->offset_a));
	read_be_int32(in, &(out->offset_b));
	read_be_int32(in, &(out->track_end));
}

static void read_nrg_etnf_element(FILE *in, NRG_ETNF_ELEMENT *out)
{
	read_be_int32(in, &(out->offset));
	read_be_int32(in, &(out->length));
	read_be_int32(in, &(out->mode));
	read_be_int32(in, &(out->start));
	read_be_int32(in, &(out->unknown));
}

static int bcd_msf_to_lbn(unsigned char mm, unsigned char ss, unsigned char ff)
{
	int r;

	r = ((mm >> 4) * 10 + (mm & 0xf)) * 60 * 75;
	r += ((ss >> 4) * 10 + (ss & 0xf)) * 75;
	r += ((ff >> 4) * 10 + (ff & 0xf));

	return r;
}

