/*
** ISO/XA treating routines
**
** Writen by Sakae Tatibana <tatibana@extra.hu>
**
** 2000, 3/6 coding start
*/

#include "binary.h"

#define ISO_XA_C
#include "iso_xa.h"

int read_iso_info(FILE *in, ISO_INFO *out);
extern "C" int open_iso_file(char *in, ISO_INFO *out);
extern "C" int close_iso_file(ISO_INFO *iso);

static void set_block_size(ISO_INFO *iso);
static void set_start(ISO_INFO *iso);

/********************************************************************
   Read ISO/XA image file and set ISO_INFO structure value

   return value is  1 (success) / 0 (fail)
********************************************************************/
read_iso_info(FILE *in, ISO_INFO *out)
{
	int i;
	size_t n;
	char *buffer;

	int pipe;
	size_t skip_size;
	LBN first_path_table;

	out->stream = in;
	
	if(ftell(in) < 0){
		pipe = 1;
		/* length don't touch */
	}else{
		pipe = 0;
		fseek(in, 0, SEEK_SET);
		n = ftell(in);
		fseek(in, 0, SEEK_END);
		out->length = ftell(in) - n;
        fseek(in, 0, SEEK_SET);
	}

	buffer = (char *)malloc(65536);
	n = fread(buffer, 1, 65536, in);

	if(memcmp(buffer+(2048*16)+1, "CD001", 5) == 0){
		out->block_size = 2048;
		out->mode = 1;
		skip_size = 0;
	}else if(memcmp(buffer+(2336*16)+1, "CD001", 5) == 0){
		/* subheader dosen't exist */
		out->block_size = 2336;
		out->mode = 1;
		skip_size = 0;
	}else if(memcmp(buffer+(2336*16)+8+1, "CD001", 5) == 0){
		/* subheader exists */
		out->block_size = 2336;
		out->mode = 2;
		skip_size = 8;
	}else if(memcmp(buffer+(2352*16)+12+4+1, "CD001", 5) == 0){
		/* subheader dosen't exist */
		out->block_size = 2352;
		if(buffer[12+3] == 1 || buffer[12+3] == 2){
			out->mode = buffer[12+3];
		}else{
			out->mode = 1;
		}
		skip_size = 12 + 4; /* sync + header */
	}else if(memcmp(buffer+(2352*16)+12+4+8+1, "CD001", 5) == 0){
		/* subheader exists */
		out->block_size = 2352;
		if(buffer[12+3] == 1 || buffer[12+3] == 2){
			out->mode = buffer[12+3];
		}else{
			out->mode = 2;
		}
		skip_size = 12 + 4 + 8; /* sync + header + subheader */
	}else{
		/* not ISO/XA image. this function can't treate */ 
		out->block_size = 32768;
		out->mode = 1;
		out->memory.block[0] = buffer;
		out->memory.num = 2;
		out->num_of_block = 0;
		return 0;
	}

	out->memory.num = (int)n / (int)out->block_size;
	for(i=0;i<out->memory.num;i++){
		out->memory.block[i] = buffer + (out->block_size * i);
	}

	if(out->block_size != 2048){
		n = (out->memory.num + 1) * out->block_size - n;
		buffer = (char *)malloc(n);
		fread(buffer, 1, n, in);
		out->memory.block[i] = buffer;
	}

	/* copy volume identifier */
	memcpy(out->volume_identifier, out->memory.block[16] + skip_size + 40, 32);
	for(i=31;i>0;i--){
		if(out->volume_identifier[i] == ' '){
			out->volume_identifier[i] = '\0';
		}else{
			break;
		}
	}

	/* check littel endian path table LBN */
	first_path_table = (LBN)le_char_array_to_int32(out->memory.block[16] + skip_size + 140);

	n = (size_t)le_char_array_to_int32(out->memory.block[16] + skip_size + 144);
	if(n && n < first_path_table){
		first_path_table = n;
	}

	/* check big endian path table LBN */
	n = (size_t)be_char_array_to_int32(out->memory.block[16] + skip_size + 148);
	if(n < first_path_table){
		first_path_table = n;
	}
	
	n = (size_t)be_char_array_to_int32(out->memory.block[16] + skip_size + 152);
	if(n && n < first_path_table){
		first_path_table = n;
	}

	/* check volume descripter set terminater */
	for(i=17;i<out->memory.num;i++){
        if((unsigned char)(out->memory.block[i][skip_size]) == 0x00){
            n = (size_t)le_char_array_to_int32(out->memory.block[i] + skip_size + 71);
            if(n && n < first_path_table){
                first_path_table = n;
            }
        }else if((unsigned char)(out->memory.block[i][skip_size]) == 0xFF){
			i += 1;
            break;
        }
    }

    for(;i<out->memory.num;i++){
        if(out->memory.block[i][skip_size] == 0x01){
            /* block[i] == path table */
            out->start = first_path_table - i;
            break;
        }else if(out->memory.block[i][skip_size] == 0x22){
            /* block[i] == file & directory descriptor */

            /* first descriptor is root directory, and
               extent LBN is corrent block address      */
            out->start = le_char_array_to_int32(out->memory.block[i] + skip_size + 2) - i;
            break;
        }else if(out->memory.block[i][skip_size] == 0x00){
            /* block[i] == padding block? */

            /* skip this block */
		}else{
            /* block[i] == unknown block */

            /* skip this block */
        }
	}

	if(pipe){
		out->num_of_block = le_char_array_to_int32(out->memory.block[16] + skip_size + 80);
	}else{
		out->num_of_block = out->length / out->block_size;
	}
	
	return 1;
}

/********************************************************************
   Open ISO/XA image file and set ISO_INFO structure value

   return value is  1 (success) / 0 (fail)
********************************************************************/
int open_iso_file(char *in, ISO_INFO *out)
{
	out->stream = fopen(in, "rb");
	if(out->stream == NULL){
		return 0;
	}

	if(read_iso_info(out->stream, out)){
		return 1;
	}else{
		close_iso_file(out);
		return 0;
	}
}

/********************************************************************
   Close ISO/XA image infomation structure

   return value is  1 (success) / 0 (fail)
********************************************************************/
int close_iso_file(ISO_INFO *iso)
{
	fclose(iso->stream);

	free(iso->memory.block[0]);
	if(iso->block_size != 2048){
		free(iso->memory.block[iso->memory.num]);
	}
	
	return 1;
}

