/*
** CDRWIN (Golden Hawk Technology) .CUE file treating module
**
** Writen by Sakae Tatibana <tatibana@extra.hu>
**
** 2000, 3/6 coding start
**
*/

#include <string.h>
#include <ctype.h>
#include "filename.h"
#include "binary.h"

#define CUE_C
#include "cue.h"

typedef struct {
	int count;
	char **items;
} CUE_COMMAND;

typedef struct {
	char path[FILENAME_MAX];
	size_t filesize;

	int offset_comp;
	int start_comp;

	LBN gap;

	dao_track *previous_track;
	dao_track *current_track;
} CUE_IMAGE;

extern "C" int open_cue_file(char *file, DAO_INFO *out);
int read_cue_info(FILE *in, DAO_INFO *out);

static int get_cue_command(FILE *in, CUE_COMMAND *out);
static int parse_line(char *in, CUE_COMMAND *out);
static void string_to_upper(char *string);
static void release_cue_command(CUE_COMMAND *command);
static int open_cue_image(char *image, char *cue, CUE_IMAGE *out);

/********************************************************************
   Open *.CUE file and set DAO_INFO structure value

   return value is  1 (success) / 0 (fail)
********************************************************************/
int open_cue_file(char *file, DAO_INFO *out)
{
	int r;
	FILE *cue;

	out->path = file;

	cue = fopen(file, "rt");
	if(cue == NULL){
		r = 0;
	}else if(!read_cue_info(cue, out)){
		r = 0;
	}else{
		r = 1;
	}

	fclose(cue);
	return r;
}

/********************************************************************
   Read CUE file and set DAO_INFO structure value

   return value is same as out->num_of_track
********************************************************************/
int read_cue_info(FILE *in, DAO_INFO *out)
{
	int file_flag, track_flag;

	CUE_IMAGE work;
	CUE_COMMAND command;

	file_flag = 0;
	track_flag = 0;

	memset(&work, 0, sizeof(CUE_IMAGE));
	
	while(get_cue_command(in, &command)){
		if(strcmp(command.items[0], "FILE") == 0){
			
			file_flag = 1;
			track_flag = 0;

			if(work.current_track){
				work.current_track->num_of_block = (work.filesize + work.offset_comp - (work.current_track->start - work.start_comp) * 2352) / work.current_track->block_size;
				work.start_comp += (int)work.current_track->num_of_block + (int)work.current_track->start;
				work.offset_comp = 0;
			}
			
			string_to_upper(command.items[2]);
			if(strcmp(command.items[2], "BINARY") != 0){
				release_cue_command(&command);
				return 0;
			}

			if(! open_cue_image(command.items[1], out->path, &work)){
				release_cue_command(&command);
				return 0;
			}

		}else if(file_flag && strcmp(command.items[0], "TRACK") == 0){

			int track;
			
			track_flag = 1;

			track = atoi(command.items[1]) - 1;
			if(track != out->num_of_track){
				release_cue_command(&command);
				return 0;
			}

			out->num_of_track += 1;
			out->track[track].index = track;

			strcpy(out->track[track].path, work.path);

			string_to_upper(command.items[2]);

			if(strcmp(command.items[2], "AUDIO") == 0){
				out->track[track].type = DAO_TRACK_TYPE_AUDIO;
				out->track[track].block_size = 2352;
			}else if(strcmp(command.items[2], "CDG") == 0){
				out->track[track].type = DAO_TRACK_TYPE_AUDIO;
				out->track[track].block_size = 2448;
			}else if(strcmp(command.items[2], "MODE1/2048") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE1;
				out->track[track].block_size = 2048;
			}else if(strcmp(command.items[2], "MODE1/2352") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE1;
				out->track[track].block_size = 2352;
			}else if(strcmp(command.items[2], "MODE2/2336") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE2;
				out->track[track].block_size = 2336;
			}else if(strcmp(command.items[2], "MODE2/2352") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE2;
				out->track[track].block_size = 2352;
			}else if(strcmp(command.items[2], "CDI/2336") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE2;
				out->track[track].block_size = 2336;
			}else if(strcmp(command.items[2], "CDI/2352") == 0){
				out->track[track].type = DAO_TRACK_TYPE_MODE2;
				out->track[track].block_size = 2352;
			}else{
				release_cue_command(&command);
				return 0;
			}

			if(work.current_track){
				work.previous_track = work.current_track;
				work.current_track = &(out->track[track]);
			}else{
                work.current_track = &(out->track[track]);
            }
			
		}else if(track_flag && strcmp(command.items[0], "PREGAP") == 0){
			
			int mm,ss,ff;

			sscanf(command.items[2], "%2d:%2d:%2d", &mm, &ss, &ff);

			work.gap = mm * 60 * 75 + ss * 75 + ff;
			
		}else if(track_flag && strcmp(command.items[0], "INDEX") == 0){

			int index;
			int mm,ss,ff;

			sscanf(command.items[2], "%2d:%2d:%2d", &mm, &ss, &ff);
			index = atoi(command.items[1]);

			if(index == 0 && work.previous_track){
				work.previous_track->num_of_block = mm * 60 * 75 + ss * 75 + ff - (work.previous_track->start - work.start_comp);
				work.offset_comp += (2352 - (int)work.previous_track->block_size) * (int)work.previous_track->num_of_block;
			}else if(index == 1){
				work.current_track->start = mm * 60 * 75 + ss * 75 + ff + work.start_comp;
				if(work.previous_track && ! work.previous_track->num_of_block){
					work.previous_track->num_of_block = work.current_track->start - work.previous_track->start;
					work.offset_comp += (2352 - (int)work.previous_track->block_size) * (int)work.previous_track->num_of_block;
				}
				work.current_track->offset = (mm * 60 * 75 + ss * 75 + ff) * 2352 - work.offset_comp;
				if(work.gap){
					work.current_track->start += work.gap;
					work.start_comp += (int)work.gap;
					work.gap = 0;
				}
			}

		}

		release_cue_command(&command);

	}

	if(work.current_track){
		work.current_track->num_of_block = (work.filesize + work.offset_comp - (work.current_track->start - work.start_comp) * 2352) / work.current_track->block_size;
	}

	return 1;
}

static int get_cue_command(FILE *in, CUE_COMMAND *out)
{
	char *line;
	const int line_max = 512;

	line = (char *)malloc(line_max);

	while(fgets(line, line_max-1, in)){
		parse_line(line, out);
		if(out->count){
			free(line);
			return out->count;
		}
	}

	free(line);
	return 0;
}

static int parse_line(char *in, CUE_COMMAND *out)
{
	int i,n;

	char *buffer;
	int length;

	int space,quote;

	length = (int)strlen(in);
	buffer = (char *)malloc(length+1);
	memcpy(buffer, in, length+1);

    out->items = (char **)malloc(sizeof(char *));

	for(out->count=0,space=1,quote=0,i=0,n=0;i<length+1;i++){
		if(!quote && !space && isspace(buffer[i]) || buffer[i] == '\0'){
			out->count += 1;
			out->items = (char **)realloc(out->items, out->count*sizeof(char **));
			out->items[out->count-1] = (char *)malloc(i-n+1);
			memcpy(out->items[out->count-1], buffer+n, i-n);
			out->items[out->count-1][i-n] = '\0';
			space = 1;
		}
        if(!quote && space && !isspace(buffer[i])){
			n = i;
			space = 0;
        }
		if(buffer[i] == '"'){
			memmove(buffer+i,buffer+i+1, length-i);
			length -= 1;
			if(space){
				space = 0;
				n = i;
			}
			quote = !quote;
			i -= 1;
		}
	}
	
	free(buffer);
	if(out->count){
		string_to_upper(out->items[0]);
	}

	return out->count;
}

static void string_to_upper(char *string)
{
	int length;
	int i;

	length = (int)strlen(string);

	for(i=0;i<length;i++){
		string[i] = (unsigned char)toupper(string[i]);
	}

	return;
}

static void release_cue_command(CUE_COMMAND *command)
{
	int i;

	for(i=0;i<command->count;i++){
		free(command->items[i]);
	}

	free(command->items);

	command->items = NULL;
	command->count = 0;

	return;
}

static int open_cue_image(char *image, char *cue, CUE_IMAGE *out)
{
	if(file_exists(image)){
		strcpy(out->path, image);
	}else{
		strcpy(out->path, cue);
		cut_filename(out->path);
		strcat(out->path, read_filename(image));
		if(! file_exists(out->path)){
			return 0;
		}
	}
	
	out->filesize = get_filesize(out->path);

	out->previous_track = NULL;
	out->current_track = NULL;

	return 1;
}