#include "Header.h"
#include "stdafx.h"
#include <algorithm>
#include <functional>
#include <sstream>
#include "CDImage.h"
#include "Utility.h"
#include <boost/crc.hpp>
#include "CDFileSystem.h"

using namespace std;

class GCMCDFileSystem::FindTargetDirectory :public unary_function<DIRECTORY, bool>
{
private:
	const string& m_strDirectoryName;
public:
	explicit FindTargetDirectory(const string& name) : m_strDirectoryName(name){}
	bool operator()(const DIRECTORY& directory)const{
		return directory.name == m_strDirectoryName;
	}
};

GCMCDFileSystem::GCMCDFileSystem(const string& file_name)
{
	m_iFileNumber = 0;
	m_pCDImage = auto_ptr<GCMCDImage>(new GCMCDImage(file_name));
	m_bOpened = false;
	if (!m_pCDImage->IsOpen()){
		return;
	}

	//m_pFile = auto_ptr<CFile>(new CFile);
	//if (!m_pFile->Open(m_pCDImage->GetTrackFilePath(0).c_str(), CFile::modeRead | CFile::shareDenyNone | CFile::osRandomAccess)){
	//	return;
	//}

	m_vecPathTable.push_back(DIRECTORY("dummy", 0));

	//{[Lqq ǂݍ
	auto_ptr<char> buffer_volume = auto_ptr<char>(new char[2048]);
	m_pCDImage->GetDataToMemory(buffer_volume.get(), 0, 16, 1);
	int path_table_size, path_table_lba, root_directory_record_size;
	memcpy(&path_table_size, buffer_volume.get() + 132, sizeof(int));
	memcpy(&path_table_lba, buffer_volume.get() + 140, sizeof(int));
	memcpy(&root_directory_record_size, buffer_volume.get() + 166, sizeof(int));

	int path_table_block_count = path_table_size / 2048 + 1;
	auto_ptr<char> buffer_path_table = auto_ptr<char>(new char[2048 * path_table_block_count]);
	m_pCDImage->GetDataToMemory(buffer_path_table.get(), 0, path_table_lba, path_table_block_count);
	int path_table_read_size = 0;
	while (path_table_read_size < path_table_size){
		DIRECTORY work(buffer_path_table.get() + path_table_read_size);
		path_table_read_size += work.data_size;
		if (work.parent > 1){
			work.name = m_vecPathTable[work.parent].name + work.name + "\\";
		} else {
			if (m_vecPathTable.size() > 1){
				work.name += "\\";
			} else if (m_vecPathTable.size() == 1){
				work.record_size = root_directory_record_size;
			}
		}
		m_vecPathTable.push_back(work);
	}
	for (vector<DIRECTORY>::iterator i = m_vecPathTable.begin() + 1; i != m_vecPathTable.end(); i++){
		int path_table_buffer_size = i->record_size / 2048 + 1;
		auto_ptr<char> buffer_directory_record = auto_ptr<char>(new char[2048 * path_table_buffer_size]);
		m_pCDImage->GetDataToMemory(buffer_directory_record.get(), 0, i->lba_start, path_table_buffer_size);
		int directory_read_size = 0;
		while (directory_read_size < i->record_size){
			FILE work(buffer_directory_record.get() + directory_read_size);
			if (work.end){
				break;
			}
			directory_read_size += work.data_size;
			if (work.is_directory){
				if (work.name != ""){
					string directory_name = GCMUtility::ToLower(i->name + work.name) + "\\";
					vector<DIRECTORY>::iterator iter = find_if(m_vecPathTable.begin(), m_vecPathTable.end(), FindTargetDirectory(directory_name));
					if (iter != m_vecPathTable.end()){
						iter->record_size = work.file_size;
					}
				}
				continue;
			}
			work.name = GCMUtility::ToLower(i->name + work.name);
			m_vecFileTable.push_back(work);
			if (work.is_last_record){
				break;
			}
		}
	}
	m_bOpened = true;
}

GCMCDFileSystem::~GCMCDFileSystem()
{
}

class FindFileInVector : unary_function<GCMCDFileSystem::FILE, bool>
{
private:
	const string& m_strFileName;
public:
	explicit FindFileInVector(const string& file_name) : m_strFileName(file_name){}
	bool operator()(const GCMCDFileSystem::FILE& file)const{
		return m_strFileName == file.name;
	}
};

int GCMCDFileSystem::FileOpen(const string& file_name)
{
	string file_name_comp = GCMUtility::ToLower(file_name);
	vector<FILE>::iterator iter_file = find_if(m_vecFileTable.begin(), m_vecFileTable.end(), FindFileInVector(file_name_comp));
	if (iter_file == m_vecFileTable.end()){
		return -1;
	}
	int file_size = iter_file->file_size;
	int sector = (file_size / 2048) + 1;
	int memory_size = sector * 2048;
	boost::shared_ptr<char> ptr = boost::shared_ptr<char>(new char[memory_size]);
	m_pCDImage->GetDataToMemory(ptr.get(), 0, iter_file->lba_start, sector);
	m_vecFilePointer.push_back(ptr);
	m_vecFileSize.push_back(file_size);
	return m_iFileNumber++;
}

const char* GCMCDFileSystem::GetFilePointer(int file_number)const
{
	if (m_iFileNumber <= file_number){
		return 0;
	}
	return m_vecFilePointer[file_number].get();
}

unsigned int GCMCDFileSystem::GetFileSize(int file_number)const
{
	if (m_iFileNumber <= file_number){
		return 0;
	}
	return m_vecFileSize[file_number];
}

const std::string GCMCDFileSystem::GetFileCRC(const std::string& file_name)
{
	int number_target = FileOpen(file_name);
	if (number_target < 0){
		return "FFFFFFFF";
	}
	const char * ptr_target = GetFilePointer(number_target);
	unsigned int file_size = GetFileSize(number_target);
	boost::crc_32_type boost_crc;
	boost_crc.process_bytes(ptr_target, file_size);
	unsigned int crc = boost_crc.checksum();
	ostringstream oss;
	oss << std::hex << crc;
	string res = GCMUtility::ToUpper(oss.str());
	if (res.size() < 8){
		res = string(8 - res.size(), '0') + res;
	}
	return res;
}

GCMCDFileSystem::DIRECTORY::DIRECTORY(const char* pointer)
{
	unsigned char name_length;
	memcpy(&name_length, pointer + 0, sizeof(unsigned char));
	data_size = 8 + name_length + ((name_length % 2 == 1) ? 1 : 0);
	memcpy(&lba_start, pointer + 2, sizeof(unsigned int));
	memcpy(&parent, pointer + 6, sizeof(unsigned short));
	auto_ptr<char> buffer_name = auto_ptr<char>(new char[name_length + 1]);
	ZeroMemory(buffer_name.get(), name_length + 1);
	memcpy(buffer_name.get(), pointer + 8, name_length);
	name = GCMUtility::ToLower(buffer_name.get());
}

GCMCDFileSystem::FILE::FILE(const char *pointer)
{
	memcpy(&data_size, pointer + 0, sizeof(unsigned char));
	if ((end = (data_size == 0)) == true){
		return;
	}
	memcpy(&lba_start, pointer + 2, sizeof(unsigned int));
	memcpy(&file_size, pointer + 10, sizeof(unsigned int));
	unsigned char flag;
	memcpy(&flag, pointer + 25, sizeof(unsigned char));
	is_last_record = (flag & 0x80) ? true : false;
	is_directory = (flag & 0x02) ? true : false;
	unsigned char name_length;
	memcpy(&name_length, pointer + 32, sizeof(unsigned char));
	auto_ptr<char> buffer_name = auto_ptr<char>(new char[name_length + 1]);
	ZeroMemory(buffer_name.get(), name_length + 1);
	memcpy(buffer_name.get(), pointer + 33, name_length);
	name = buffer_name.get();
	size_t index = name.find(';');
	if (index != string::npos){
		name.erase(index);
	}
	if (!name.empty()){
		if (name[0] == 1){
			name.clear();
		}
	}
}
